/*************************************************************************
 *                                                                       *
 *   B H C C y l   V 1 . 0                                  Sep. 2006    *
 *                                                                       *
 *   Reference: Bohren, Huffman,                                         *
 *              Absorption and Scattering of Light by Small Particles,   *
 *              Wiley-Intersciencen, 1983                                *
 *                                                                       *
 *   Author:    Sven Biermann                                            *
 *   Contact:   www.SvenBiermann.de                                      *
 *                                                                       *
 *************************************************************************/

#include <iostream>
#include <iomanip>
#include "numerical.h"
using namespace std;

void CalcCyl(const double x, const cmplx m, const double *theta, const long nang, cmplx* T1, cmplx* T2);
void showHelp();

int main(int argc, char *argv[], char *envp[])
{
	const double pi_div_180 = 3.1415926535897932385/180.0;
	if (argc<6)  showHelp();
	int argvOffset = 1;
	double x = 0.0;  // Mie parameter
	cmplx refrel(0.0);  // relative refractive index
	if (strcmp(argv[1], "-d") == 0)
	{
		if (argc<9)  showHelp();
		double diameter   = atof(argv[2]);    // diameter of the sphere im m
		if (diameter <= 0.0)  showHelp();
		double wavelength = atof(argv[3]);    // wavelength in m
		if (wavelength <= 0.0)  showHelp();
		double refmed     = atof(argv[4]);    // refmed = (real) refractive index of surrounding medium
		if (refmed <= 0.0)  showHelp();
		double realrefsph = atof(argv[5]);    // real part of refractive index of the sphere
		if (realrefsph <= 0.0)  showHelp();
		cmplx  refsph(realrefsph, 0.0);       // refractive index of the sphere = real + i imaginary
		refrel = refsph/refmed;        // relative refractive index
		x = 3.1415926535897932385*diameter*refmed / wavelength;  // Mie parameter
		argvOffset = 6;
	}
	else
	{
  	x = atof(argv[1]);
		if (x <= 0.0)  showHelp();
		refrel.SetRe(atof(argv[2]));
		if (refrel.GetRe() <= 0.0)  showHelp();
		argvOffset = 3;
	}
	long   nang   = atol(argv[argvOffset]);   // number of angles between "FromDeg" and "ToDeg"
  if (nang <= 0)  showHelp();
	double theta1 = atof(argv[argvOffset+1]); // FromDeg
	if (theta1 < 0.0  ||  theta1 > 180.0)  showHelp();
	double theta2 = atof(argv[argvOffset+2]); // ToDeg
	if (theta2 < 0.0  ||  theta2 > 180.0)  showHelp();
	double dTheta = (theta2-theta1)*pi_div_180;  // convert to radian
  theta1 *= pi_div_180;
  argvOffset += 3;
	int dataType = 0;  // calculate intensities as standard option
	if (argvOffset  < argc-1)  showHelp();  // too many command line options
	if (argvOffset == argc-1)
	{
    if (strcmp(argv[argvOffset],"-a") == 0)
			dataType = 1;  // calculate scattering amplitudes
		else if (strcmp(argv[argvOffset],"-b") == 0)
			dataType = 2;  // calculate scattering amplitudes and intensities
		else if (strcmp(argv[argvOffset],"-c") == 0)
			dataType = 3;  // calculate scattering amplitudes, intensities, and phases
		else showHelp();
	}

	double *theta = new double[nang+1];  // Angles for CalcMieForGivenAngles(...)
	for (long j=1; j<=nang; j++)  theta[j] = theta1 + double(j-1)/double(nang-1)*dTheta;

	cmplx *T1, *T2;                // Scattering Amplitudes
	T1 = new cmplx[nang+1];        // Allocating Memory for T1 and T2  (CalcMieForGivenAngles(...))
	T2 = new cmplx[nang+1];

	cout << "Mie Scattering for  x = " << setprecision(32) << x << ",  m = " << refrel << endl;
	CalcCyl(x, refrel, theta, nang, T1, T2);

	switch(dataType)
	{
	case 1:
	  cout << "Angle\tRe(T1)\tIm(T1)\tRe(T2)\tIm(T2)" << endl;
	  for (long j=1; j<=nang; j++)   // j<2*nang for CalcMie(...)
		  cout << theta[j] / pi_div_180 << "\t" << T1[j].GetRe() << "\t" << T1[j].GetIm() << "\t"
			                                      << T2[j].GetRe() << "\t" << T2[j].GetIm() << endl;
		break;
	case 2:
	  cout << "Angle\tRe(T1)\tIm(T1)\t|T1|^2\tRe(T2)\tIm(T2)\t|T2|^2" << endl;
	  for (long j=1; j<=nang; j++)   // j<2*nang for CalcMie(...)
		  cout << theta[j] / pi_div_180 << "\t"
			     << T1[j].GetRe() << "\t" << T1[j].GetIm() << "\t" << T1[j].AbsSqr() << "\t"
			     << T2[j].GetRe() << "\t" << T2[j].GetIm() << "\t" << T2[j].AbsSqr() << endl;
		break;
	case 3:
	  cout << "Angle\tRe(T1)\tIm(T1)\t|T1|^2\targ(T1)\tRe(T2)\tIm(T2)\t|T2|^2\targ(T2)" << endl;
	  for (long j=1; j<=nang; j++)   // j<2*nang for CalcMie(...)
		  cout << theta[j] / pi_div_180 << "\t"
			     << T1[j].GetRe() << "\t" << T1[j].GetIm() << "\t"
					 << T1[j].AbsSqr() << "\t" << T1[j].GetPhi() << "\t"
			     << T2[j].GetRe() << "\t" << T2[j].GetIm() << "\t"
					 << T2[j].AbsSqr() << "\t" << T2[j].GetPhi() << endl;
		break;
	default:
	  cout << "Angle\t|T1|^2\t|T2|^2" << endl;
	  for (long j=1; j<=nang; j++)   // j<2*nang for CalcMie(...)
		  cout << theta[j] / pi_div_180 << "\t" << T1[j].AbsSqr() << "\t" << T2[j].AbsSqr() << endl;
	}

	return 0;
}


void CalcCyl(const double x, const cmplx m, const double *theta, const long nang, cmplx* T1, cmplx* T2)
{
	cmplx an, bn, a0, b0;

	const cmplx y = m*x;
	const long nstop = long(x + 4.0*pow(x, 1.0/3.0) + 3.0);   // series terminated after nstop terms
	long nmx = long(y.GetAbs());
	if (nstop>nmx)  nmx = nstop;
	nmx += 16;
	cmplx *D  = new cmplx[nmx];

	// logarithmic derivative G[k] calculated by downward recurrence
	// beginning with initial value (0.0+i*0.0) at k = nmx
	D[nmx-1] = cmplx(0.0, 0.0);
  for (long n = nmx-1; n>0; n--)
		D[n-1] = ((n-1.0)/y) - (1.0 / (n/y + D[n]));

	// Bessel functions J[n] computed by downward recurrence
	// beginning at n = nstop + ndelta
	// Bessel functions Y[n] computed by upward recurrence
	// Def.: BJ[n+1] := J[n],  BY[n+1] := Y[n]
	long ndelta = long(pow(101.0 + x, 0.499));
  long mst = nstop + ndelta;
	mst -= mst % 2;  // mst must be even
	cmplx  *H = new cmplx[mst+1];
	double *J = new double[mst+1];
	double *Y = new double[mst+1];
	double *F = new double[mst+1];

	F[mst+1] = 0.0;
	F[mst]   = 1.0E-32;
	for (long n = mst; n > 0; n--)
		F[n-1] = 2.0*double(n)*F[n]/x - F[n+1];
	double alpha = 0.0;
	for (long n = mst-2; n>0; n -= 2)
		alpha += F[n];
	alpha *= 2.0;
	alpha += F[0];
	for (long n=0; n<mst; n++)
		J[n] = F[n] / alpha;

	Y[0] = 0.0;
	for (long n = mst/2-1; n > 0; n--)
		if (n % 2)
		  Y[0] += J[2*n]/double(n);
		else
		  Y[0] -= J[2*n]/double(n);
	Y[0] *= 2.0;
	Y[0] += J[0] * (log(x/2.0) + 0.577215664);  // 0.577... is Euler's constant
	Y[0] *= 0.636619772;   //  ... / pi/2
	Y[1] = (J[1]*Y[0] - 0.636619772/x) / J[0];
	for (long n = 1; n<nstop; n++)
		Y[n+1] = 2.0*double(n)*Y[n]/x - Y[n-1];
	for (long n = 0; n<=nstop; n++)
		H[n] = cmplx(J[n], Y[n]);
//for (long n = 0; n<=nstop; n++)  cout << "D[" << n <<"] = " << D[n] << "\tJ[" << n <<"] = " << J[n] << "\tY[" << n <<"] = " << Y[n] << endl;
	
	a0 = (D[0]*J[0]/m + J[1])  /  (D[0]*H[0]/m + H[1]);
	b0 = (D[0]*J[0]*m + J[1])  /  (D[0]*H[0]*m + H[1]);
//cout << "a[0] = " << a0 << "\tb[0] = " << b0 << endl;
	for (long j=1; j<=nang; j++)
	{
		T1[j] = b0;
		T2[j] = a0;
	}
	for (long n=1; n<=nstop; n++)
	{
		an = ((D[n]/m + n/x)*J[n] - J[n-1])  /  ((D[n]/m + n/x)*H[n] - H[n-1]);
		bn = ((D[n]*m + n/x)*J[n] - J[n-1])  /  ((D[n]*m + n/x)*H[n] - H[n-1]);
//cout << "a[" << n << "] = " << an << "\tb[" << n << "] = " << bn << endl;
		for (long j=1; j<=nang; j++)
		{
			double c = cos(n*theta[j]);
			T1[j] += 2.0 * bn * c;
			T2[j] += 2.0 * an * c;
		}
	}
}


void showHelp()
{
	cout << "Too few, too many, or wrong command line options!" << endl;
	cout << "BHCCyl calculates light scattering by a cylinder for orthog. incidence." << endl;
  cout << "Usage:" << endl;
	cout << "  BHCCyl x m nAngles FromDeg ToDeg       or" << endl;
	cout << "  BHCCyl -d diameter wavelength nmed nsphere nAngles FromDeg ToDeg" << endl;
	cout << "where x is the Mie parameter, m is the (real) relative refractive index," << endl;
	cout << "nAngleg is the number of angles to be calculated between FromDeg and" << endl;
	cout << "ToDeg. You have to specify real values for each parameter except for" << endl;
	cout << "the number of angles which has to be an integer." << endl;
	cout << "The option -d as first parameter indicates that you specify the" << endl;
	cout << "diameter and (vacuum) wavelength in microns, the refractive index of" << endl;
	cout << "the medium and that of the cylinder, respectively, instead of x and m." << endl;
	cout << "These parameters have to be real numbers." << endl;
	cout << "Further options may follow the above parameter list:"<< endl;
	cout << "  -a    calculates complex scattering amplitudes instead of intensities" << endl;
	cout << "  -b    calculates both" << endl;
	cout << "  -c    calculates both and additionally the phases (in rad)" << endl;
	cout << "BHCCyl [parameters] >> Filename.txt    saves the output to a file" << endl;
	cout << "named \"FileName.txt\". If you want to use spaces within that name" << endl;
	cout << "you have to enclose it with quotation marks." << endl;
	cout << "Examples:" << endl;
	cout << "  BHCCyl 1203.5 0.75 1280 0.0 30.5" << endl;
	cout << "  BHCCyl -d 950.0 0.5145 1.333 1.0 1024 33.5 35.5 -b >> scattering.txt" << endl;
	exit(0);
}
